<?php
session_start();
require_once __DIR__ . '/db.php';

// Require login and a valid checklist id
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

try {
    $pdo = getPDO();
} catch (Throwable $e) {
    die('Database connection error');
}

$checklistId = isset($_GET['checklist_id']) ? (int)$_GET['checklist_id'] : 0;
$busNumber = isset($_GET['bus_number']) ? trim($_GET['bus_number']) : '';
if ($checklistId <= 0) {
    header('Location: entry.php');
    exit;
}

// Handle submit: save results
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $statuses = isset($_POST['status']) && is_array($_POST['status']) ? $_POST['status'] : [];
    try {
        $pdo->beginTransaction();
        // Idempotent: clear existing results for this checklist
        $stmtDel = $pdo->prepare('DELETE FROM checklist_results WHERE checklist_id = ?');
        $stmtDel->execute([$checklistId]);

        $stmtIns = $pdo->prepare('INSERT INTO checklist_results (checklist_id, item_id, status) VALUES (?, ?, ?)');
        foreach ($statuses as $itemId => $status) {
            $status = in_array($status, ['Pass', 'Fail', 'Not Checked'], true) ? $status : 'Not Checked';
            $stmtIns->execute([$checklistId, (int)$itemId, $status]);
        }
        $pdo->commit();
        header('Location: succes.php');
        exit;
    } catch (Throwable $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        die('Failed to save checklist results.');
    }
}

// Load items for both pages
$items = $pdo->query('SELECT item_id, category, check_type, check_item_text, standard_text FROM check_items ORDER BY check_type, category, item_id')->fetchAll();
$preOpItems = array_filter($items, fn($r) => $r['check_type'] === 'Pre-operation');
$testRunItems = array_filter($items, fn($r) => $r['check_type'] === 'Test-run');

// Load existing checklist results for this checklist_id
$existingResults = [];
try {
    $stmt = $pdo->prepare('SELECT item_id, status FROM checklist_results WHERE checklist_id = ?');
    $stmt->execute([$checklistId]);
    $results = $stmt->fetchAll();
    foreach ($results as $result) {
        $existingResults[$result['item_id']] = $result['status'];
    }
} catch (Throwable $e) {
    // If no existing results, continue with empty array
    $existingResults = [];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Update Commercial Vehicle Checklist</title>
    <!-- Favicon -->
    <link rel="icon" type="image/png" href="images/logo1.png">
    <link rel="shortcut icon" type="image/png" href="images/fav.png">

    <style>
        /* General body styling */
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            background: linear-gradient(rgba(35, 43, 56, 0.6), rgba(35, 43, 56, 0.6)), url('images/slider01.png') no-repeat center center;
            background-size: cover; /* This makes the image cover the background */
            background-attachment: fixed; /* Keeps background fixed when scrolling */ /* Dark blue background */
            color: #333;
        }

         /* New wrapper for the main content */
         .page-wrapper {
             display: flex;
             flex-direction: column;
             align-items: center;
             padding: 25px;
             gap: 25px; /* Space between header and form */
             padding-bottom: 50px; /* Add space above footer */
         }

         /* --- Header Styling --- */
         .header-main {
             width: 100%;
             max-width: 1100px;
             background: #fff;
             padding: 10px 30px;
             border-radius: 15px;
             box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
             display: flex;
             justify-content: space-between;
             align-items: center;
             box-sizing: border-box;
         }
         .header-logo img {
             height: 40px;
         }
         .header-nav {
             display: flex;
             align-items: center;
             gap: 25px;
         }
         .header-logout-btn {
             background-color: #dc3545;
             color: white;
             border: none;
             padding: 8px 16px;
             border-radius: 5px;
             font-size: 14px;
             font-weight: 500;
             cursor: pointer;
             text-decoration: none;
             transition: background-color 0.3s ease;
         }
         .header-logout-btn:hover {
             background-color: #c82333;
         }

         /* Main container for the form */
        .checklist-container {
            width: 100%;
            max-width: 1100px;
            background: #fff;
            padding: 30px 40px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
            border-radius: 20px;
            box-sizing: border-box;
        }
        
        /* Title styling */
        h2 {
            text-align: center;
            color: #444;
            font-size: 1.4em;
            margin-bottom: 5px;
            text-transform: uppercase;
        }

        /* Subtitle styling */
        .subtitle {
            text-align: center;
            margin-top: 0;
            margin-bottom: 25px;
            color: #777;
            font-size: 0.9em;
        }

        /* Update notice */
        .update-notice {
            background: linear-gradient(135deg, #28a745, #20c997);
            color: white;
            padding: 15px;
            border-radius: 10px;
            text-align: center;
            margin-bottom: 25px;
            font-weight: 500;
            box-shadow: 0 2px 10px rgba(40, 167, 69, 0.3);
        }
        
        /* Table styling */
        .checklist-table {
            width: 100%;
            border-collapse: collapse;
            font-size: 14px;
        }

        .checklist-table th,
        .checklist-table td {
            border: 1px solid #e0e0e0;
            padding: 12px;
            text-align: left;
            vertical-align: middle;
        }

        .checklist-table thead th {
            background-color: #f8f9fa;
            font-weight: bold;
            text-align: center;
            color: #555;
        }
        
        .pass-fail-col {
            width: 60px;
            text-align: center;
        }

        /* --- Custom Checkbox Styling --- */
        .custom-checkbox {
            display: inline-block;
            width: 20px;
            height: 20px;
            background-color: #fff;
            border: 1px solid #ccc;
            border-radius: 3px;
            cursor: pointer;
            position: relative;
        }
        .custom-checkbox.ticked {
            background-color: #4CAF50;
            border-color: #4CAF50;
        }
        .custom-checkbox.ticked::after {
            content: '✔';
            color: white;
            position: absolute;
            left: 50%;
            top: 50%;
            transform: translate(-50%, -50%);
            font-size: 14px;
        }

        /* Hiding tables by default */
        .checklist-page {
            display: none;
        }
        .checklist-page.active {
            display: block;
        }
        
        /* --- Button Container & Styling --- */
        .button-container {
            display: flex;
            justify-content: flex-end;
            gap: 15px;
            margin-top: 30px;
        }
        .btn {
            color: white;
            border: none;
            padding: 12px 30px;
            font-size: 16px;
            font-weight: bold;
            border-radius: 8px;
            cursor: pointer;
            transition: background-color 0.3s ease;
        }
        .btn-next { background-color: #ff9800; }
        .btn-next:hover { background-color: #e68900; }
        .btn-back { background-color: #6c757d; }
        .btn-back:hover { background-color: #5a6268; }
        .btn-submit { background-color: #9ccc65; }
        .btn-submit:hover { background-color: #8bc34a; }

        /* --- New Footer Styling --- */
        .site-footer {
            width: 100%;
            background-color: #0c1e35;
            padding: 18px 0;
            color: #a0b0c0; /* Light, muted blue/grey */
            text-align: center;
            font-size: 0.8em;
            letter-spacing: 1px;
            text-transform: uppercase;
        }

    </style>
</head>
<body>

<?php include 'include/header.php'; ?>


<div class="page-wrapper">
    

    <div class="checklist-container">
        <div class="update-notice">
             Updating Checklist for Bus: <strong><?php echo htmlspecialchars($busNumber); ?></strong>
        </div>
        
        <form id="vehicleChecklistForm" method="POST" action="<?php echo 'update_table.php?checklist_id=' . (int)$checklistId . '&bus_number=' . urlencode($busNumber); ?>">
            <div id="page1" class="checklist-page active">
                <h2>Daily Pre-operation Checklist</h2>
                <p class="subtitle">Visual Inspection (Before Starting the Engine)</p>
                <table class="checklist-table" id="preOpTable">
                    <thead><tr><th>Category</th><th>Check Item</th><th>Recommended Status / Standard</th><th class="pass-fail-col">Pass</th><th class="pass-fail-col">Fail</th></tr></thead>
                    <tbody>
                        <?php 
                        $groupedItems = [];
                        foreach ($preOpItems as $row) {
                            $groupedItems[$row['category']][] = $row;
                        }
                        foreach ($groupedItems as $category => $items): ?>
                            <?php foreach ($items as $index => $row): ?>
                                <tr>
                                    <?php if ($index === 0): ?>
                                        <td rowspan="<?php echo count($items); ?>"><?php echo htmlspecialchars($category); ?></td>
                                    <?php endif; ?>
                                    <td><?php echo htmlspecialchars($row['check_item_text']); ?></td>
                                    <td><?php echo htmlspecialchars($row['standard_text']); ?></td>
                                    <td class="pass-fail-col">
                                        <span class="custom-checkbox <?php echo (isset($existingResults[$row['item_id']]) && $existingResults[$row['item_id']] === 'Pass') ? 'ticked' : ''; ?>" 
                                              data-item-id="<?php echo (int)$row['item_id']; ?>" 
                                              data-status="Pass"></span>
                                    </td>
                                    <td class="pass-fail-col">
                                        <span class="custom-checkbox <?php echo (isset($existingResults[$row['item_id']]) && $existingResults[$row['item_id']] === 'Fail') ? 'ticked' : ''; ?>" 
                                              data-item-id="<?php echo (int)$row['item_id']; ?>" 
                                              data-status="Fail"></span>
                                        <input type="hidden" name="status[<?php echo (int)$row['item_id']; ?>]" 
                                               value="<?php echo isset($existingResults[$row['item_id']]) ? htmlspecialchars($existingResults[$row['item_id']]) : 'Not Checked'; ?>">
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <div id="page2" class="checklist-page">
                <h2>Test Run Inspection</h2>
                <p class="subtitle">Engine On & During Short Drive</p>
                <table class="checklist-table" id="testRunTable">
                     <thead><tr><th>Category</th><th>Check Item</th><th>Recommended status / Standard</th><th class="pass-fail-col">Pass</th><th class="pass-fail-col">Fail</th></tr></thead>
                    <tbody>
                        <?php 
                        $groupedItems = [];
                        foreach ($testRunItems as $row) {
                            $groupedItems[$row['category']][] = $row;
                        }
                        foreach ($groupedItems as $category => $items): ?>
                            <?php foreach ($items as $index => $row): ?>
                                <tr>
                                    <?php if ($index === 0): ?>
                                        <td rowspan="<?php echo count($items); ?>"><?php echo htmlspecialchars($category); ?></td>
                                    <?php endif; ?>
                                    <td><?php echo htmlspecialchars($row['check_item_text']); ?></td>
                                    <td><?php echo htmlspecialchars($row['standard_text']); ?></td>
                                    <td class="pass-fail-col">
                                        <span class="custom-checkbox <?php echo (isset($existingResults[$row['item_id']]) && $existingResults[$row['item_id']] === 'Pass') ? 'ticked' : ''; ?>" 
                                              data-item-id="<?php echo (int)$row['item_id']; ?>" 
                                              data-status="Pass"></span>
                                    </td>
                                    <td class="pass-fail-col">
                                        <span class="custom-checkbox <?php echo (isset($existingResults[$row['item_id']]) && $existingResults[$row['item_id']] === 'Fail') ? 'ticked' : ''; ?>" 
                                              data-item-id="<?php echo (int)$row['item_id']; ?>" 
                                              data-status="Fail"></span>
                                        <input type="hidden" name="status[<?php echo (int)$row['item_id']; ?>]" 
                                               value="<?php echo isset($existingResults[$row['item_id']]) ? htmlspecialchars($existingResults[$row['item_id']]) : 'Not Checked'; ?>">
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <div class="button-container">
                <button type="button" id="backBtn" class="btn btn-back" style="display: none;">Back</button>
                <button type="button" id="nextBtn" class="btn btn-next">Next</button>
                <button type="submit" id="submitBtn" class="btn btn-submit" style="display: none;">Update Checklist</button>
            </div>
        </form>
    </div>
</div>

<footer class="site-footer">
    <p>COPYRIGHT &copy; 2025 SLTDIGITAL</p>
</footer>


<script>
    document.addEventListener('DOMContentLoaded', () => {
        const page1 = document.getElementById('page1');
        const page2 = document.getElementById('page2');
        const nextBtn = document.getElementById('nextBtn');
        const backBtn = document.getElementById('backBtn');
        const submitBtn = document.getElementById('submitBtn');
        const form = document.getElementById('vehicleChecklistForm');

        // --- NAVIGATION LOGIC ---
        nextBtn.addEventListener('click', () => {
            page1.classList.remove('active');
            page2.classList.add('active');
            nextBtn.style.display = 'none';
            backBtn.style.display = 'inline-block';
            submitBtn.style.display = 'inline-block';
        });

        backBtn.addEventListener('click', () => {
            page2.classList.remove('active');
            page1.classList.add('active');
            nextBtn.style.display = 'inline-block';
            backBtn.style.display = 'none';
            submitBtn.style.display = 'none';
        });

        // --- CHECKBOX INTERACTIVITY wired to hidden inputs (no schema change) ---
        const checkboxes = document.querySelectorAll('.custom-checkbox');
        checkboxes.forEach(box => {
            box.addEventListener('click', function() {
                const itemId = this.getAttribute('data-item-id');
                const intendedStatus = this.getAttribute('data-status');
                const parentRow = this.closest('tr');
                const boxesInRow = parentRow.querySelectorAll('.custom-checkbox');
                boxesInRow.forEach(b => b.classList.remove('ticked'));
                this.classList.add('ticked');
                const hiddenInput = parentRow.querySelector(`input[type="hidden"][name="status[${itemId}]"]`);
                if (hiddenInput) {
                    hiddenInput.value = intendedStatus;
                }
            });
        });

        // Allow normal POST submit to server which persists results
        // (client JS no longer intercepts submit)
    });
</script>

</body>
</html>
