<?php
session_start();
require_once __DIR__ . '/db.php';

// Require login and a valid checklist id
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

try {
    $pdo = getPDO();
} catch (Throwable $e) {
    die('Database connection error');
}

$checklistId = isset($_GET['checklist_id']) ? (int)$_GET['checklist_id'] : 0;
$busNumber = isset($_GET['bus_number']) ? trim($_GET['bus_number']) : '';
$isNewChecklist = isset($_GET['new_checklist']) && $_GET['new_checklist'] == '1';

// If no checklist_id and not a new checklist, redirect to entry
if ($checklistId <= 0 && !$isNewChecklist) {
    header('Location: entry.php');
    exit;
}

// If it's a new checklist, we don't have a checklist_id yet
if ($isNewChecklist && $checklistId <= 0) {
    $checklistId = 0; // Will be created after successful submission
}

// Load items for both pages (moved before POST handling)
$items = $pdo->query('SELECT item_id, category, check_type, check_item_text, standard_text FROM check_items ORDER BY check_type, category, item_id')->fetchAll();
$preOpItems = array_filter($items, fn($r) => $r['check_type'] === 'Pre-operation');
$testRunItems = array_filter($items, fn($r) => $r['check_type'] === 'Test-run');

// Handle submit: save results
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $statuses = isset($_POST['status']) && is_array($_POST['status']) ? $_POST['status'] : [];
    
    // Validation: Check if all items are filled (not "Not Checked")
    $allItemsFilled = true;
    $unfilledItems = [];
    $totalItems = count($items);
    $filledItems = 0;
    
    // Debug: Check what we received
    error_log("Total items: " . $totalItems);
    error_log("Statuses received: " . print_r($statuses, true));
    
    foreach ($items as $item) {
        $itemId = $item['item_id'];
        $status = isset($statuses[$itemId]) ? $statuses[$itemId] : 'Not Checked';
        
        if ($status === 'Not Checked') {
            $allItemsFilled = false;
            $unfilledItems[] = $item['check_item_text'];
        } else {
            $filledItems++;
        }
    }
    
    error_log("Filled items: " . $filledItems . " out of " . $totalItems);
    error_log("All items filled: " . ($allItemsFilled ? 'Yes' : 'No'));
    
    if (!$allItemsFilled) {
        // Set validation error message
        $validationError = 'Please fill all checklist items before submitting. You have filled ' . $filledItems . ' out of ' . $totalItems . ' items.';
    } else {
        // All items are filled, proceed with submission
        try {
            $pdo->beginTransaction();
            
            // If this is a new checklist, create the main record first
            if ($isNewChecklist && $checklistId <= 0) {
                $stmtMain = $pdo->prepare('INSERT INTO checklist_main (bus_number, user_id, check_date) VALUES (?, ?, CURDATE())');
                $stmtMain->execute([$busNumber, $_SESSION['user_id']]);
                $checklistId = (int)$pdo->lastInsertId();
            }
            
            // Clear existing results for this checklist (in case of update)
            $stmtDel = $pdo->prepare('DELETE FROM checklist_results WHERE checklist_id = ?');
            $stmtDel->execute([$checklistId]);

            // Insert new results
            $stmtIns = $pdo->prepare('INSERT INTO checklist_results (checklist_id, item_id, status) VALUES (?, ?, ?)');
            foreach ($statuses as $itemId => $status) {
                $status = in_array($status, ['Pass', 'Fail', 'Not Checked'], true) ? $status : 'Not Checked';
                $stmtIns->execute([$checklistId, (int)$itemId, $status]);
            }
            $pdo->commit();
            header('Location: succes.php');
            exit;
        } catch (Throwable $e) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            die('Failed to save checklist results.');
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Commercial Vehicle Checklists</title>

    <!-- Favicon -->
    <link rel="icon" type="image/png" href="images/logo1.png">
    <link rel="shortcut icon" type="image/png" href="images/fav.png">
    
    <!-- SweetAlert2 CSS -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <style>
        /* General body styling */
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            background: linear-gradient(rgba(35, 43, 56, 0.6), rgba(35, 43, 56, 0.6)), url('images/slider01.png') no-repeat center center;
            background-size: cover; /* This makes the image cover the background */
            background-attachment: fixed; /* Keeps background fixed when scrolling */ /* Dark blue background */
            color: #333;
        }

         /* New wrapper for the main content */
         .page-wrapper {
             display: flex;
             flex-direction: column;
             align-items: center;
             padding: 25px;
             gap: 25px; /* Space between header and form */
             padding-bottom: 50px; /* Add space above footer */
         }

         /* --- Header Styling --- */
         .header-main {
             width: 100%;
             max-width: 1100px;
             background: #fff;
             padding: 10px 30px;
             border-radius: 15px;
             box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
             display: flex;
             justify-content: space-between;
             align-items: center;
             box-sizing: border-box;
         }
         .header-logo img {
             height: 40px;
         }
         .header-nav {
             display: flex;
             align-items: center;
             gap: 25px;
         }
         .header-logout-btn {
             background-color: #dc3545;
             color: white;
             border: none;
             padding: 8px 16px;
             border-radius: 5px;
             font-size: 14px;
             font-weight: 500;
             cursor: pointer;
             text-decoration: none;
             transition: background-color 0.3s ease;
         }
         .header-logout-btn:hover {
             background-color: #c82333;
         }

         /* Main container for the form */
        .checklist-container {
            width: 100%;
            max-width: 1100px;
            background: #fff;
            padding: 30px 40px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
            border-radius: 20px;
            box-sizing: border-box;
        }
        
        /* Title styling */
        h2 {
            text-align: center;
            color: #444;
            font-size: 1.4em;
            margin-bottom: 5px;
            text-transform: uppercase;
        }

        /* Subtitle styling */
        .subtitle {
            text-align: center;
            margin-top: 0;
            margin-bottom: 25px;
            color: #777;
            font-size: 0.9em;
        }
        
        /* Table styling */
        .checklist-table {
            width: 100%;
            border-collapse: collapse;
            font-size: 14px;
        }

        .checklist-table th,
        .checklist-table td {
            border: 1px solid #e0e0e0;
            padding: 12px;
            text-align: left;
            vertical-align: middle;
        }

        .checklist-table thead th {
            background-color: #f8f9fa;
            font-weight: bold;
            text-align: center;
            color: #555;
        }
        
        .pass-fail-col {
            width: 60px;
            text-align: center;
        }

        /* --- Custom Checkbox Styling --- */
        .custom-checkbox {
            display: inline-block;
            width: 20px;
            height: 20px;
            background-color: #fff;
            border: 1px solid #ccc;
            border-radius: 3px;
            cursor: pointer;
            position: relative;
        }
        .custom-checkbox.ticked {
            background-color: #4CAF50;
            border-color: #4CAF50;
        }
        .custom-checkbox.ticked::after {
            content: '✔';
            color: white;
            position: absolute;
            left: 50%;
            top: 50%;
            transform: translate(-50%, -50%);
            font-size: 14px;
        }

        /* Hiding tables by default */
        .checklist-page {
            display: none;
        }
        .checklist-page.active {
            display: block;
        }
        
        /* --- Button Container & Styling --- */
        .button-container {
            display: flex;
            justify-content: flex-end;
            gap: 15px;
            margin-top: 30px;
        }
        .btn {
            color: white;
            border: none;
            padding: 12px 30px;
            font-size: 16px;
            font-weight: bold;
            border-radius: 8px;
            cursor: pointer;
            transition: background-color 0.3s ease;
        }
        .btn-next { background-color: #ff9800; }
        .btn-next:hover { background-color: #e68900; }
        .btn-back { background-color: #6c757d; }
        .btn-back:hover { background-color: #5a6268; }
        .btn-submit { background-color: #9ccc65; }
        .btn-submit:hover { background-color: #8bc34a; }

        /* --- Back Button Styling --- */
        .back-button-container {
            display: flex;
            justify-content: flex-start;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 1px solid #e0e0e0;
        }

        .back-button {
            background: #6c757d;
            color: white;
            text-decoration: none;
            padding: 10px 20px;
            border-radius: 8px;
            font-size: 0.9rem;
            font-weight: 500;
            transition: background-color 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 5px;
        }

        .back-button:hover {
            background: #5a6268;
            color: white;
            text-decoration: none;
        }

        /* --- New Footer Styling --- */
        .site-footer {
            width: 100%;
            background-color: #0c1e35;
            padding: 18px 0;
            color: #a0b0c0; /* Light, muted blue/grey */
            text-align: center;
            font-size: 0.8em;
            letter-spacing: 1px;
            text-transform: uppercase;
        }

        /* --- SweetAlert Custom Styling --- */
        .swal2-popup {
            font-family: Arial, sans-serif !important;
            border-radius: 15px !important;
            width: 350px !important;
            padding: 20px !important;
        }
        
        .swal2-title {
            font-family: Arial, sans-serif !important;
            font-weight: 600 !important;
            font-size: 18px !important;
            margin-bottom: 10px !important;
        }
        
        .swal2-content {
            font-family: Arial, sans-serif !important;
            font-size: 14px !important;
            margin-bottom: 15px !important;
        }
        
        .swal2-actions {
            margin-top: 15px !important;
        }
        
        .swal2-confirm {
            font-size: 14px !important;
            padding: 8px 20px !important;
            background-color: #ff9800 !important;
            border: none !important;
        }

        /* --- Responsive Design --- */
        @media (max-width: 768px) {
            .back-button-container {
                justify-content: center;
            }
            
            .back-button {
                font-size: 0.8rem;
                padding: 8px 16px;
            }
        }

    </style>
</head>
<body>

<?php include 'include/header.php'; ?>


<div class="page-wrapper">
    
    <div class="checklist-container">
        <!-- <div class="back-button-container">
            <a href="entry.php" class="back-button">← Back to Entry</a>
        </div> -->
        <form id="vehicleChecklistForm" method="POST" action="<?php echo 'table.php?bus_number=' . urlencode($busNumber) . ($isNewChecklist ? '&new_checklist=1' : '&checklist_id=' . (int)$checklistId); ?>">
            <div id="page1" class="checklist-page active">
                <h2>Daily Pre-operation Checklist</h2>
                <p class="subtitle">Visual Inspection (Before Starting the Engine)</p>
                <table class="checklist-table" id="preOpTable">
                    <thead><tr><th>Category</th><th>Check Item</th><th>Recommended Status / Standard</th><th class="pass-fail-col">Pass</th><th class="pass-fail-col">Fail</th></tr></thead>
                    <tbody>
                        <?php 
                        $groupedItems = [];
                        foreach ($preOpItems as $row) {
                            $groupedItems[$row['category']][] = $row;
                        }
                        foreach ($groupedItems as $category => $items): ?>
                            <?php foreach ($items as $index => $row): ?>
                                <tr>
                                    <?php if ($index === 0): ?>
                                        <td rowspan="<?php echo count($items); ?>"><?php echo htmlspecialchars($category); ?></td>
                                    <?php endif; ?>
                                    <td><?php echo htmlspecialchars($row['check_item_text']); ?></td>
                                    <td><?php echo htmlspecialchars($row['standard_text']); ?></td>
                                    <td class="pass-fail-col">
                                        <span class="custom-checkbox" data-item-id="<?php echo (int)$row['item_id']; ?>" data-status="Pass"></span>
                                    </td>
                                    <td class="pass-fail-col">
                                        <span class="custom-checkbox" data-item-id="<?php echo (int)$row['item_id']; ?>" data-status="Fail"></span>
                                        <input type="hidden" name="status[<?php echo (int)$row['item_id']; ?>]" value="Not Checked">
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <div id="page2" class="checklist-page">
                <h2>Test Run Inspection</h2>
                <p class="subtitle">Engine On & During Short Drive</p>
                <table class="checklist-table" id="testRunTable">
                     <thead><tr><th>Category</th><th>Check Item</th><th>Recommended status / Standard</th><th class="pass-fail-col">Pass</th><th class="pass-fail-col">Fail</th></tr></thead>
                    <tbody>
                        <?php 
                        $groupedItems = [];
                        foreach ($testRunItems as $row) {
                            $groupedItems[$row['category']][] = $row;
                        }
                        foreach ($groupedItems as $category => $items): ?>
                            <?php foreach ($items as $index => $row): ?>
                                <tr>
                                    <?php if ($index === 0): ?>
                                        <td rowspan="<?php echo count($items); ?>"><?php echo htmlspecialchars($category); ?></td>
                                    <?php endif; ?>
                                    <td><?php echo htmlspecialchars($row['check_item_text']); ?></td>
                                    <td><?php echo htmlspecialchars($row['standard_text']); ?></td>
                                    <td class="pass-fail-col">
                                        <span class="custom-checkbox" data-item-id="<?php echo (int)$row['item_id']; ?>" data-status="Pass"></span>
                                    </td>
                                    <td class="pass-fail-col">
                                        <span class="custom-checkbox" data-item-id="<?php echo (int)$row['item_id']; ?>" data-status="Fail"></span>
                                        <input type="hidden" name="status[<?php echo (int)$row['item_id']; ?>]" value="Not Checked">
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <div class="button-container">
                <button type="button" id="backBtn" class="btn btn-back" style="display: none;">Back</button>
                <button type="button" id="nextBtn" class="btn btn-next">Next</button>
                <button type="submit" id="submitBtn" class="btn btn-submit" style="display: none;">Submit</button>
            </div>
        </form>
    </div>
</div>

<footer class="site-footer">
    <p>COPYRIGHT &copy; 2025 SLTDIGITAL</p>
</footer>


<script>
    document.addEventListener('DOMContentLoaded', () => {
        const page1 = document.getElementById('page1');
        const page2 = document.getElementById('page2');
        const nextBtn = document.getElementById('nextBtn');
        const backBtn = document.getElementById('backBtn');
        const submitBtn = document.getElementById('submitBtn');
        const form = document.getElementById('vehicleChecklistForm');

        // --- NAVIGATION LOGIC ---
        nextBtn.addEventListener('click', () => {
            page1.classList.remove('active');
            page2.classList.add('active');
            nextBtn.style.display = 'none';
            backBtn.style.display = 'inline-block';
            submitBtn.style.display = 'inline-block';
        });

        backBtn.addEventListener('click', () => {
            page2.classList.remove('active');
            page1.classList.add('active');
            nextBtn.style.display = 'inline-block';
            backBtn.style.display = 'none';
            submitBtn.style.display = 'none';
        });

        // --- CHECKBOX INTERACTIVITY wired to hidden inputs (no schema change) ---
        const checkboxes = document.querySelectorAll('.custom-checkbox');
        checkboxes.forEach(box => {
            box.addEventListener('click', function() {
                const itemId = this.getAttribute('data-item-id');
                const intendedStatus = this.getAttribute('data-status');
                const parentRow = this.closest('tr');
                const boxesInRow = parentRow.querySelectorAll('.custom-checkbox');
                boxesInRow.forEach(b => b.classList.remove('ticked'));
                this.classList.add('ticked');
                const hiddenInput = parentRow.querySelector(`input[type="hidden"][name="status[${itemId}]"]`);
                if (hiddenInput) {
                    hiddenInput.value = intendedStatus;
                }
            });
        });

        // Add form submission validation
        form.addEventListener('submit', function(e) {
            // Check if all items are filled
            const allCheckboxes = document.querySelectorAll('.custom-checkbox');
            const allHiddenInputs = document.querySelectorAll('input[type="hidden"][name^="status"]');
            let unfilledCount = 0;
            
            allHiddenInputs.forEach(input => {
                if (input.value === 'Not Checked') {
                    unfilledCount++;
                }
            });
            
            if (unfilledCount > 0) {
                e.preventDefault(); // Prevent form submission
                
                Swal.fire({
                    icon: 'warning',
                    title: 'Incomplete Checklist!',
                    text: `Please fill all checklist items before submitting. You have ${unfilledCount} unchecked items remaining.`,
                    confirmButtonColor: '#ff9800',
                    confirmButtonText: 'OK',
                    customClass: {
                        popup: 'swal-custom-popup'
                    }
                });
                
                return false;
            }
        });
    });

    <?php if (isset($validationError)): ?>
    // Display SweetAlert for validation error
    Swal.fire({
        icon: 'warning',
        title: 'Incomplete Checklist!',
        text: '<?php echo addslashes($validationError); ?>',
        confirmButtonColor: '#ff9800',
        confirmButtonText: 'OK',
        customClass: {
            popup: 'swal-custom-popup'
        }
    });
    <?php endif; ?>
</script>

</body>
</html>

