<?php
session_start();
require_once __DIR__ . '/db.php';

// Redirect to login if not authenticated
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$successMessage = '';
$errorMessage = '';
$sweetAlertMessage = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $busNumber = isset($_POST['busNumber']) ? trim($_POST['busNumber']) : '';

    if ($busNumber === '') {
        $errorMessage = 'Bus number is required.';
    } else {
        try {
            $pdo = getPDO();
            
            // Check if bus number already exists in the database
            $checkStmt = $pdo->prepare('SELECT COUNT(*) FROM checklist_main WHERE bus_number = ?');
            $checkStmt->execute([$busNumber]);
            $busExists = $checkStmt->fetchColumn() > 0;
            
            if ($busExists) {
                $sweetAlertMessage = 'Warning: Bus number "' . htmlspecialchars($busNumber) . '" already recorded.';
            } else {
                // Bus number doesn't exist, proceed to table without creating database record
                // The database record will be created only after successful checklist submission
                header('Location: table.php?bus_number=' . urlencode($busNumber) . '&new_checklist=1');
                exit;
            }
        } catch (Throwable $e) {
            $errorMessage = 'Error: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>MMC Kottawa - Bus Entry</title>
    
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600&display=swap" rel="stylesheet">
    
    <!-- Favicon -->
    <link rel="icon" type="image/png" href="images/logo1.png">
    <link rel="shortcut icon" type="image/png" href="images/fav.png">
    
    <!-- SweetAlert2 CSS -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

    <style>
        /* --- Root Variables --- */
        :root {
            --gradient-top: #F8FFED; /* Light Green */
            --gradient-middle: #FFFFFF; /* White */
            --gradient-bottom: #FFF8E8; /* Light Cream */
            --input-border: #ddd;
            --input-focus-border: #007BFF;
            --input-shadow-focus: rgba(0, 123, 255, 0.2);
            --button-start: #FFD200; /* Yellow/Orange */
            --button-end: #FF9F00; /* Darker Orange */
            --text-color-dark: #333;
            --text-color-light: #777;
            --white: #ffffff;
            --card-shadow: 0 10px 30px rgba(0, 0, 0, 0.15);
            --footer-bg: #232B38; /* Dark background for footer */
            --footer-text: #B0B0B0; /* Light grey text for footer */
        }

        /* --- Global Reset & Body Styles --- */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Poppins', sans-serif;
            color: var(--text-color-dark);
            min-height: 100vh; /* Full viewport height */
            display: flex;
            flex-direction: column; /* Column layout to push footer to bottom */
            justify-content: space-between; /* Space out content and footer */
            align-items: center; /* Center horizontally */
            
            /* --- Background Image Setup --- */
            /* ❗ Replace with your background image path */
            background: linear-gradient(rgba(35, 43, 56, 0.6), rgba(35, 43, 56, 0.6)), url('images/slider01.png') no-repeat center center;
            background-size: cover; /* This makes the image cover the background */
            background-attachment: fixed; /* Keeps background fixed when scrolling */
        }

        /* --- Main Login Container (Card) --- */
        .login-container {
            flex-grow: 1; /* Allows container to take available space, pushing footer down */
            display: flex;
            justify-content: center;
            align-items: center;
            width: 100%;
            padding: 20px; /* Padding around the card */
            margin-top: 2px;
            margin-bottom: 20px;
        }

        .card-content {
            background: linear-gradient(to bottom, var(--gradient-top), var(--gradient-middle), var(--gradient-bottom));
            width: 100%;
            max-width: 400px; /* Reduced width for smaller card */
            padding: 35px 30px; /* Reduced padding for more compact size */
            border-radius: 15px; /* Rounded corners for the card */
            box-shadow: var(--card-shadow);
            text-align: center;
        }

        /* --- Logo --- */
        .logo-container img {
            max-width: 180px; /* Reduced logo size */
            margin-bottom: 25px; /* Reduced space below logo */
        }

        /* --- Input Group --- */
        .input-group {
            margin-bottom: 20px; /* Reduced space below input */
        }

        .input-group input {
            width: 90%; /* Adjusted width */
            padding: 15px 20px; /* Slightly more padding */
            border: 1px solid var(--input-border);
            border-radius: 8px;
            font-size: 1.05rem; /* Slightly larger font */
            font-family: 'Poppins', sans-serif;
            text-align: center; /* Placeholder text centered */
            transition: border-color 0.3s, box-shadow 0.3s;
        }

        .input-group input::placeholder {
            color: #aaa;
        }

        .input-group input:focus {
            outline: none;
            border-color: var(--input-focus-border);
            box-shadow: 0 0 0 3px var(--input-shadow-focus);
        }

        /* --- Next Button --- */
        .next-button {
            width: 90%; /* Aligned with input */
            padding: 15px;
            border: none;
            border-radius: 8px;
            background: linear-gradient(90deg, var(--button-start), var(--button-end));
            color: var(--white);
            font-size: 1.2rem;
            font-weight: 600;
            cursor: pointer;
            box-shadow: 0 5px 15px rgba(255, 160, 0, 0.3); /* Button shadow */
            transition: transform 0.2s, box-shadow 0.2s;
        }

        .next-button:hover {
            transform: translateY(-2px);
            box-shadow: 0 7px 20px rgba(255, 160, 0, 0.4);
        }

        /* --- Footer --- */
        .footer {
            width: 100%;
            background-color: var(--footer-bg);
            color: var(--footer-text);
            text-align: center;
            padding: 15px 0;
            font-size: 0.85rem;
            margin-top: auto; /* Pushes footer to the bottom */
        }

        /* --- Accessibility: Visually hidden but readable for screen readers --- */
        .sr-only {
            position: absolute;
            width: 1px;
            height: 1px;
            padding: 0;
            margin: -1px;
            overflow: hidden;
            clip: rect(0, 0, 0, 0);
            white-space: nowrap;
            border: 0;
        }
        
        /* --- SweetAlert Custom Styling --- */
        .swal-custom-popup {
            font-family: 'Poppins', sans-serif !important;
        }
        
        .swal2-popup {
            font-family: 'Poppins', sans-serif !important;
            border-radius: 15px !important;
            width: 350px !important; /* Reduced width */
            padding: 20px !important; /* Reduced padding */
        }
        
        .swal2-title {
            font-family: 'Poppins', sans-serif !important;
            font-weight: 600 !important;
            font-size: 18px !important; /* Smaller title */
            margin-bottom: 10px !important; /* Reduced margin */
        }
        
        .swal2-content {
            font-family: 'Poppins', sans-serif !important;
            font-size: 14px !important; /* Smaller text */
            margin-bottom: 15px !important; /* Reduced margin */
        }
        
        .swal2-actions {
            margin-top: 15px !important; /* Reduced margin */
        }
        
        .swal2-confirm {
            font-size: 14px !important; /* Smaller button text */
            padding: 8px 20px !important; /* Smaller button padding */
        }

        /* --- Responsive Adjustments --- */
        @media (max-width: 600px) {
            .swal2-popup {
                width: 300px !important; /* Even smaller on mobile */
                padding: 15px !important;
            }
            
            .swal2-title {
                font-size: 16px !important; /* Smaller title on mobile */
            }
            
            .swal2-content {
                font-size: 13px !important; /* Smaller text on mobile */
            }
            
            .card-content {
                max-width: 95%; /* Adjust for smaller screens */
                padding: 40px 25px;
                border-radius: 10px;
            }

            .logo-container img {
                max-width: 180px;
                margin-bottom: 30px;
            }

            .input-group input,
            .next-button {
                width: 100%; /* Full width on very small screens */
                padding: 14px;
                font-size: 1rem;
            }

            .footer {
                padding: 12px 0;
                font-size: 0.8rem;
            }
        }
    </style>
</head>
<body>

    <?php include 'include/header.php'; ?>

    <div class="login-container">
        <div class="card-content">
            <div class="logo-container">
                <img src="images/logo1.png" alt="MMC Kottawa Logo">
            </div>
            
            <?php if ($successMessage !== ''): ?>
                <div style="margin-bottom: 15px; color: #155724; background: #d4edda; border: 1px solid #c3e6cb; padding: 10px; border-radius: 6px;">
                    <?php echo htmlspecialchars($successMessage); ?>
                </div>
            <?php endif; ?>
            <?php if ($errorMessage !== ''): ?>
                <div style="margin-bottom: 15px; color: #721c24; background: #f8d7da; border: 1px solid #f5c6cb; padding: 10px; border-radius: 6px;">
                    <?php echo htmlspecialchars($errorMessage); ?>
                </div>
            <?php endif; ?>

            <form action="entry.php" method="POST">
                <div class="input-group">
                    <label for="busNumber" class="sr-only">Enter the bus number</label>
                    <input type="text" id="busNumber" name="busNumber" placeholder="Enter the bus number" required>
                </div>

                <button type="submit" class="next-button">Next</button>
            </form>
        </div>
    </div>

    <footer class="footer">
        <p>COPYRIGHT © 2025 SLTDIGITAL</p>
    </footer>

    <script>
        <?php if ($sweetAlertMessage !== ''): ?>
        // Display SweetAlert for warning message
        Swal.fire({
            icon: 'warning',
            title: 'Bus Number Already Exists!',
            text: '<?php echo addslashes($sweetAlertMessage); ?>',
            confirmButtonColor: '#FF9F00',
            confirmButtonText: 'OK',
            customClass: {
                popup: 'swal-custom-popup'
            }
        });
        <?php endif; ?>
    </script>

</body>
</html>