<?php
session_start();
require_once __DIR__ . '/db.php';

// Redirect to login if not authenticated
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$submittedBuses = [];
$errorMessage = '';
$successMessage = '';
$sweetAlertSuccess = '';

// Handle delete action
if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
    try {
        $pdo = getPDO();
        $deleteId = (int)$_GET['id'];
        
        // Verify the record belongs to the current user
        $checkStmt = $pdo->prepare('SELECT user_id FROM checklist_main WHERE id = ?');
        $checkStmt->execute([$deleteId]);
        $record = $checkStmt->fetch();
        
        if ($record && $record['user_id'] == $_SESSION['user_id']) {
            // Delete the record
            $deleteStmt = $pdo->prepare('DELETE FROM checklist_main WHERE id = ?');
            $deleteStmt->execute([$deleteId]);
            $sweetAlertSuccess = 'Bus record deleted successfully!';
        } else {
            $errorMessage = 'You are not authorized to delete this record.';
        }
    } catch (Throwable $e) {
        $errorMessage = 'Error deleting record: ' . $e->getMessage();
    }
}

try {
    $pdo = getPDO();
    
    // Get all submitted bus numbers for the current user
    $stmt = $pdo->prepare('
        SELECT 
            id,
            bus_number, 
            check_date
        FROM checklist_main 
        WHERE user_id = ? 
        ORDER BY check_date DESC, id DESC
    ');
    $stmt->execute([$_SESSION['user_id']]);
    $submittedBuses = $stmt->fetchAll();
    
} catch (Throwable $e) {
    $errorMessage = 'Error loading submitted buses: ' . $e->getMessage();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>MMC Kottawa - Already Submitted</title>
    
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600&display=swap" rel="stylesheet">
    
    <!-- Favicon -->
    <link rel="icon" type="image/png" href="images/logo1.png">
    <link rel="shortcut icon" type="image/png" href="images/fav.png">
    
    <!-- SweetAlert2 CSS -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

    <style>
        /* --- Root Variables --- */
        :root {
            --gradient-top: #F8FFED;
            --gradient-middle: #FFFFFF;
            --gradient-bottom: #FFF8E8;
            --primary-blue: #007BFF;
            --primary-blue-dark: #0056b3;
            --text-color-dark: #333;
            --text-color-light: #777;
            --border-color: #ddd;
            --white: #ffffff;
            --card-shadow: 0 10px 30px rgba(0, 0, 0, 0.15);
            --footer-bg: #232B38;
            --footer-text: #B0B0B0;
            --table-header-bg: #f8f9fa;
            --table-border: #dee2e6;
        }

        /* --- Global Reset & Body Styles --- */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Poppins', sans-serif;
            color: var(--text-color-dark);
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            background: linear-gradient(rgba(35, 43, 56, 0.6), rgba(35, 43, 56, 0.6)), url('images/slider01.png') no-repeat center center;
            background-size: cover;
            background-attachment: fixed;
        }

        /* --- Main Container --- */
        .main-container {
            flex-grow: 1;
            display: flex;
            justify-content: center;
            align-items: flex-start;
            width: 100%;
            padding: 20px;
            margin-top: 20px;
        }

        .content-wrapper {
            background: linear-gradient(to bottom, var(--gradient-top), var(--gradient-middle), var(--gradient-bottom));
            width: 100%;
            max-width: 900px;
            padding: 30px;
            border-radius: 15px;
            box-shadow: var(--card-shadow);
        }

        /* --- Page Title --- */
        .page-title {
            text-align: center;
            margin-bottom: 30px;
        }

        .page-title h1 {
            font-size: 2rem;
            font-weight: 600;
            color: var(--text-color-dark);
            margin-bottom: 10px;
        }

        .page-title p {
            color: var(--text-color-light);
            font-size: 1rem;
        }

        /* --- Back Button Container --- */
        .back-button-container {
            display: flex;
            justify-content: flex-end;
            margin-top: 30px;
        }

        .back-button {
            background: var(--primary-blue);
            color: var(--white);
            text-decoration: none;
            padding: 10px 20px;
            border-radius: 8px;
            font-size: 0.9rem;
            font-weight: 500;
            transition: background-color 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 5px;
        }

        .back-button:hover {
            background: var(--primary-blue-dark);
            color: var(--white);
            text-decoration: none;
        }

        /* --- Table Styles --- */
        .table-container {
            overflow-x: auto;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }

        .submitted-table {
            width: 100%;
            border-collapse: collapse;
            background: var(--white);
            font-size: 0.95rem;
        }

        .submitted-table th {
            background: var(--table-header-bg);
            color: var(--text-color-dark);
            font-weight: 600;
            padding: 15px 12px;
            text-align: left;
            border-bottom: 2px solid var(--table-border);
            font-size: 0.9rem;
        }

        .submitted-table th:last-child {
            text-align: center;
        }

        .submitted-table td {
            padding: 12px;
            border-bottom: 1px solid var(--table-border);
            vertical-align: middle;
        }

        .submitted-table td:last-child {
            text-align: center;
        }

        .submitted-table tbody tr:hover {
            background-color: #f8f9fa;
        }

        .submitted-table tbody tr:last-child td {
            border-bottom: none;
        }

        /* --- Bus Number Styling --- */
        .bus-number {
            font-weight: 600;
            color: var(--primary-blue);
            font-size: 1.1rem;
        }

        /* --- Date Styling --- */
        .date-cell {
            color: var(--text-color-light);
            font-size: 0.9rem;
        }

        /* --- Action Buttons --- */
        .action-buttons {
            display: flex;
            gap: 6px;
            justify-content: center;
        }

        .update-btn {
            background: #28a745;
            color: var(--white);
            border: none;
            padding: 6px 12px;
            border-radius: 5px;
            font-size: 0.8rem;
            font-weight: 500;
            cursor: pointer;
            text-decoration: none;
            transition: background-color 0.3s ease;
        }

        .update-btn:hover {
            background: #218838;
        }

        .delete-btn {
            background: #dc3545;
            color: var(--white);
            border: none;
            padding: 6px 12px;
            border-radius: 5px;
            font-size: 0.8rem;
            font-weight: 500;
            cursor: pointer;
            text-decoration: none;
            transition: background-color 0.3s ease;
        }

        .delete-btn:hover {
            background: #c82333;
        }

        /* --- Empty State --- */
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: var(--text-color-light);
        }

        .empty-state h3 {
            font-size: 1.2rem;
            margin-bottom: 10px;
            color: var(--text-color-dark);
        }

        .empty-state p {
            font-size: 0.95rem;
        }

        /* --- Error Message --- */
        .error-message {
            background-color: rgba(220, 53, 69, 0.08);
            color: #dc3545;
            border: 1px solid rgba(220, 53, 69, 0.35);
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            text-align: center;
        }

        /* --- Success Message --- */
        .success-message {
            background-color: rgba(40, 167, 69, 0.08);
            color: #155724;
            border: 1px solid rgba(40, 167, 69, 0.35);
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            text-align: center;
        }

        /* --- SweetAlert Custom Styling --- */
        .swal2-popup {
            font-family: 'Poppins', sans-serif !important;
            border-radius: 15px !important;
            width: 350px !important;
            padding: 20px !important;
        }
        
        .swal2-title {
            font-family: 'Poppins', sans-serif !important;
            font-weight: 600 !important;
            font-size: 18px !important;
            margin-bottom: 10px !important;
        }
        
        .swal2-content {
            font-family: 'Poppins', sans-serif !important;
            font-size: 14px !important;
            margin-bottom: 15px !important;
        }
        
        .swal2-actions {
            margin-top: 15px !important;
        }
        
        .swal2-confirm {
            font-size: 14px !important;
            padding: 8px 20px !important;
            background-color: #dc3545 !important;
            border: none !important;
        }
        
        .swal2-cancel {
            font-size: 14px !important;
            padding: 8px 20px !important;
            background-color: #6c757d !important;
            border: none !important;
        }

        /* --- Footer --- */
        .footer {
            width: 100%;
            background-color: var(--footer-bg);
            color: var(--footer-text);
            text-align: center;
            padding: 15px 0;
            font-size: 0.85rem;
            margin-top: auto;
        }

        /* --- Responsive Design --- */
        @media (max-width: 768px) {
            .content-wrapper {
                max-width: 95%;
                padding: 20px;
            }

            .back-button-container {
                justify-content: center;
            }

            .page-title h1 {
                font-size: 1.5rem;
            }

            .submitted-table {
                font-size: 0.85rem;
            }

            .submitted-table th,
            .submitted-table td {
                padding: 8px 6px;
            }

            .action-buttons {
                flex-direction: column;
                gap: 4px;
            }

            .update-btn,
            .delete-btn {
                font-size: 0.75rem;
                padding: 4px 8px;
            }
        }

        @media (max-width: 480px) {
            .main-container {
                padding: 10px;
            }

            .content-wrapper {
                padding: 15px;
            }

            .submitted-table th,
            .submitted-table td {
                padding: 6px 4px;
                font-size: 0.8rem;
            }
        }
    </style>
</head>
<body>

    <?php include 'include/header.php'; ?>

    <div class="main-container">
        <div class="content-wrapper">
            <div class="page-title">
                <h1>Already Submitted Buses</h1>
                <p>Your submitted bus checklist records</p>
            </div>

            <?php if ($errorMessage !== ''): ?>
                <div class="error-message">
                    <?php echo htmlspecialchars($errorMessage); ?>
                </div>
            <?php endif; ?>

            <?php if ($successMessage !== ''): ?>
                <div class="success-message">
                    <?php echo htmlspecialchars($successMessage); ?>
                </div>
            <?php endif; ?>

            <?php if (empty($submittedBuses)): ?>
                <div class="empty-state">
                    <h3>No Submitted Buses</h3>
                    <p>You haven't submitted any bus checklists yet.</p>
                </div>
            <?php else: ?>
                <div class="table-container">
                    <table class="submitted-table">
                        <thead>
                            <tr>
                                <th>Bus Number</th>
                                <th>Check Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($submittedBuses as $bus): ?>
                                <tr>
                                    <td>
                                        <span class="bus-number"><?php echo htmlspecialchars($bus['bus_number']); ?></span>
                                    </td>
                                    <td class="date-cell">
                                        <?php echo date('M d, Y', strtotime($bus['check_date'])); ?>
                                    </td>
                                    <td>
                                        <div class="action-buttons">
                                            <a href="update_table.php?checklist_id=<?php echo $bus['id']; ?>&bus_number=<?php echo urlencode($bus['bus_number']); ?>" class="update-btn">
                                                Update
                                            </a>
                                            <a href="#" 
                                               class="delete-btn" 
                                               onclick="confirmDelete(<?php echo $bus['id']; ?>, '<?php echo htmlspecialchars($bus['bus_number']); ?>')">
                                                Delete
                                            </a>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
            
            <div class="back-button-container">
                <a href="entry.php" class="back-button">← Back</a>
            </div>
        </div>
    </div>

    <footer class="footer">
        <p>COPYRIGHT © 2025 SLTDIGITAL</p>
    </footer>

    <script>
        function confirmDelete(busId, busNumber) {
            Swal.fire({
                title: 'Delete Bus Record?',
                text: `Are you sure you want to delete the record for bus "${busNumber}"? This action cannot be undone.`,
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#dc3545',
                cancelButtonColor: '#6c757d',
                confirmButtonText: 'Yes, Delete!',
                cancelButtonText: 'Cancel',
                customClass: {
                    popup: 'swal-custom-popup'
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    // Redirect to delete URL
                    window.location.href = `?action=delete&id=${busId}`;
                }
            });
        }

        <?php if ($sweetAlertSuccess !== ''): ?>
        // Display SweetAlert for success message
        Swal.fire({
            icon: 'success',
            title: 'Success!',
            text: '<?php echo addslashes($sweetAlertSuccess); ?>',
            confirmButtonColor: '#28a745',
            confirmButtonText: 'OK',
            customClass: {
                popup: 'swal-custom-popup'
            }
        });
        <?php endif; ?>
    </script>

</body>
</html>
