<?php
// Include database connection
require_once 'config/database.php';

// Set content type to JSON
header('Content-Type: application/json');

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Get the safeguard ID from POST data
$id = isset($_POST['id']) ? (int)$_POST['id'] : 0;

if ($id <= 0) {
    echo json_encode(['success' => false, 'message' => 'Invalid safeguard ID']);
    exit;
}

try {
    $pdo = getConnection();
    $pdo->beginTransaction();

    // First, get the safeguard document to check if it has a file
    $stmt = $pdo->prepare('SELECT document_path FROM safeguard WHERE id = ?');
    $stmt->execute([$id]);
    $safeguard = $stmt->fetch();

    if (!$safeguard) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Safeguard document not found']);
        exit;
    }

    // Delete the safeguard record
    $deleteStmt = $pdo->prepare('DELETE FROM safeguard WHERE id = ?');
    $result = $deleteStmt->execute([$id]);

    if (!$result) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Failed to delete safeguard document']);
        exit;
    }

    // If there was a document file, try to delete it from the filesystem
    if (!empty($safeguard['document_path'])) {
        $documentPath = $safeguard['document_path'];
        
        // Check if it's a relative path and convert to absolute path
        if (!preg_match('#^(?:https?://|/)#', $documentPath)) {
            $absolutePath = __DIR__ . '/../' . ltrim($documentPath, '/');
            
            // Only delete if the file exists and is within our uploads directory
            if (file_exists($absolutePath) && strpos($absolutePath, __DIR__ . '/../assets/uploads/safeguard/') === 0) {
                @unlink($absolutePath);
            }
        }
    }

    $pdo->commit();
    
    echo json_encode([
        'success' => true, 
        'message' => 'Safeguard document deleted successfully'
    ]);

} catch (PDOException $e) {
    if ($pdo && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    error_log('Database error in deleteSafeguard.php: ' . $e->getMessage());
    echo json_encode([
        'success' => false, 
        'message' => 'Database error occurred while deleting the safeguard document'
    ]);
} catch (Exception $e) {
    if ($pdo && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    error_log('General error in deleteSafeguard.php: ' . $e->getMessage());
    echo json_encode([
        'success' => false, 
        'message' => 'An unexpected error occurred'
    ]);
}
?>
